// GuiUtils - some static GUI utilities
//
// Copyright (C) 1996 by Jef Poskanzer <jef@acme.com>.  All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//
// Visit the ACME Labs Java page for up-to-date versions of this and other
// fine Java utilities: http://www.acme.com/java/

package Acme;

import java.awt.*;
import java.awt.image.*;
import java.applet.*;

/// Some static GUI utilities.
// <P>
// <A HREF="/resources/classes/Acme/GuiUtils.java">Fetch the software.</A><BR>
// <A HREF="/resources/classes/Acme.tar.Z">Fetch the entire Acme package.</A>

public class GuiUtils
    {

    /// Returns the top available parent of a component.
    public static Component getTopParent( Component comp )
	{
	Component parent;
	while ( ( parent = comp.getParent() ) != null )
	    comp = parent;
	return comp;
	}

    /// Calls pack() on the nearest parent that's a Window.
    public static void packWindow( Component comp )
	{
	while ( comp != null )
	    {
	    if ( comp instanceof Window )
		{
		((Window) comp).pack();
		return;
		}
	    comp = comp.getParent();
	    }
	}


    private static final int SPLINE_THRESH = 3;

    /// Draw a three-point spline.
    public static void drawSpline( Graphics graphics, int x1, int y1, int x2, int y2, int x3, int y3 )
	{
	int xa, ya, xb, yb, xc, yc, xp, yp;

	xa = ( x1 + x2 ) / 2;
	ya = ( y1 + y2 ) / 2;
	xc = ( x2 + x3 ) / 2;
	yc = ( y2 + y3 ) / 2;
	xb = ( xa + xc ) / 2;
	yb = ( ya + yc ) / 2;

	xp = ( x1 + xb ) / 2;
	yp = ( y1 + yb ) / 2;
	if ( Math.abs( xa - xp ) + Math.abs( ya - yp ) > SPLINE_THRESH )
	    drawSpline( graphics, x1, y1, xa, ya, xb, yb );
	else
	    graphics.drawLine( x1, y1, xb, yb );

	xp = ( x3 + xb ) / 2;
	yp = ( y3 + yb ) / 2;
	if ( Math.abs( xc - xp ) + Math.abs( yc - yp ) > SPLINE_THRESH )
	    drawSpline( graphics, xb, yb, xc, yc, x3, y3 );
	else
	    graphics.drawLine( xb, yb, x3, y3 );
	}


    private static final int DDA_SCALE = 8192;

    /// Draw a thick line.
    public static void drawThickLine( Graphics graphics, int x1, int y1, int x2, int y2, int linewidth )
	{
	// Draw the starting point filled.
	graphics.fillOval(
	    x1 - linewidth / 2, y1 - linewidth / 2, linewidth, linewidth );

	// Short-circuit zero-length lines.
	if ( x1 == x2 && y1 == y2 )
	    return;

	/* Draw, using a simple DDA. */
	if ( Math.abs( x2 - x1 ) > Math.abs( y2 - y1 ) )
	    {
	    // Loop over X domain.
	    int dy, srow;
	    int dx, col, row, prevrow;

	    if ( x2 > x1 )
		dx = 1;
	    else
		dx = -1;
	    dy = ( y2 - y1 ) * DDA_SCALE / Math.abs( x2 - x1 );
	    prevrow = row = y1;
	    srow = row * DDA_SCALE + DDA_SCALE / 2;
	    col = x1;
	    for (;;)
		{
		if ( row != prevrow )
		    {
		    graphics.drawOval(
			col - linewidth / 2, prevrow - linewidth / 2,
			linewidth, linewidth );
		    prevrow = row;
		    }
		graphics.drawOval(
		    col - linewidth / 2, row - linewidth / 2,
		    linewidth, linewidth );
		if ( col == x2 )
		    break;
		srow += dy;
		row = srow / DDA_SCALE;
		col += dx;
		}
	    }
	else
	    {
	    // Loop over Y domain.
	    int dx, scol;
	    int dy, col, row, prevcol;

	    if ( y2 > y1 )
		dy = 1;
	    else
		dy = -1;
	    dx = ( x2 - x1 ) * DDA_SCALE / Math.abs( y2 - y1 );
	    row = y1;
	    prevcol = col = x1;
	    scol = col * DDA_SCALE + DDA_SCALE / 2;
	    for ( ; ; )
		{
		if ( col != prevcol )
		    {
		    graphics.drawOval(
			prevcol - linewidth / 2, row - linewidth / 2,
			linewidth, linewidth );
		    prevcol = col;
		    }
		graphics.drawOval(
		    col - linewidth / 2, row - linewidth / 2,
		    linewidth, linewidth );
		if ( row == y2 )
		    break;
		row += dy;
		scol += dx;
		col = scol / DDA_SCALE;
		}
	    }
	}


    /// Parse a color string into a Color.  The color can be specified
    // by name as one of:
    // <BLOCKQUOTE>
    // black blue cyan darkGray gray green lightGray
    // magenta orange pink red white yellow
    // </BLOCKQUOTE>
    // Or, as an #rrggbb hex number, like in Netscape.
    public static Color parseColor( String str )
	{
	if ( str.startsWith( "#" ) )
	    {
	    try
		{
		int h = Integer.parseInt( str.substring( 1 ), 16 );
		return new Color(
		    ( h >>> 16 ) & 0xff, ( h >>> 8 ) & 0xff, h & 0xff );
		}
	    catch ( NumberFormatException e )
		{
		return null;
		}
	    }
	Color color;
	color = parseNamedColor( str );
	if ( color != null )
	    return color;
	if ( str.substring( 0, 4 ).equalsIgnoreCase( "dark" ) )
	    {
	    color = parseNamedColor( str.substring( 4 ) );
	    if ( color != null )
		return color.darker();
	    }
	if ( str.substring( 0, 5 ).equalsIgnoreCase( "light" ) )
	    {
	    color = parseNamedColor( str.substring( 5 ) );
	    if ( color != null )
		return color.brighter();
	    }
	if ( str.substring( 0, 6 ).equalsIgnoreCase( "bright" ) )
	    {
	    color = parseNamedColor( str.substring( 6 ) );
	    if ( color != null )
		return color.brighter();
	    }
	return null;
	}

    private static Color parseNamedColor( String str )
	{
	if ( str.equalsIgnoreCase( "black" ) )
	    return Color.black;
	if ( str.equalsIgnoreCase( "blue" ) )
	    return Color.blue;
	if ( str.equalsIgnoreCase( "cyan" ) )
	    return Color.cyan;
	if ( str.equalsIgnoreCase( "darkGray" ) )
	    return Color.darkGray;
	if ( str.equalsIgnoreCase( "gray" ) )
	    return Color.gray;
	if ( str.equalsIgnoreCase( "green" ) )
	    return Color.green;
	if ( str.equalsIgnoreCase( "lightGray" ) )
	    return Color.lightGray;
	if ( str.equalsIgnoreCase( "magenta" ) )
	    return Color.magenta;
	if ( str.equalsIgnoreCase( "orange" ) )
	    return Color.orange;
	if ( str.equalsIgnoreCase( "pink" ) )
	    return Color.pink;
	if ( str.equalsIgnoreCase( "red" ) )
	    return Color.red;
	if ( str.equalsIgnoreCase( "white" ) )
	    return Color.white;
	if ( str.equalsIgnoreCase( "yellow" ) )
	    return Color.yellow;
	return null;
	}


    /// Handle the standard BGCOLOR parameter.  Call as:
    // <BLOCKQUOTE>
    // Acme.GuiUtils.handleBgcolor( this );
    // </BLOCKQUOTE>
    // at the start of your init() method.
    public static void handleBgcolor( Applet applet )
	{
	String param = applet.getParameter( "bgcolor" );
	if ( param != null )
	    {
	    Color color = parseColor( param );
	    if ( color != null )
		applet.setBackground( color );
	    }
	}


    // Standard icons.

    private final static int noticeWidth = 8;
    private final static int noticeHeight = 21;
    private static int[] noticePixels = {	// color model is AARRGGBB
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff
	};
    private static ImageProducer noticeSource = null;

    public static Image noticeIcon( Component comp )
	{
	if ( noticeSource == null )
	    noticeSource = new MemoryImageSource(
		noticeWidth, noticeHeight, noticePixels, 0, noticeWidth );
	return comp.createImage( noticeSource );
	}
    
    /// Pale yellow.
    public static final Color noticeColor = new Color( 0xff, 0xff, 0x90 );


    private final static int warningWidth = 6;
    private final static int warningHeight = 19;
    private static int[] warningPixels = {	// color model is AARRGGBB
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff
	};
    private static ImageProducer warningSource = null;

    public static Image warningIcon( Component comp )
	{
	if ( warningSource == null )
	    warningSource = new MemoryImageSource(
		warningWidth, warningHeight, warningPixels, 0, warningWidth );
	return comp.createImage( warningSource );
	}
    
    /// Medium orange.
    public static final Color warningColor = new Color( 0xff, 0xa0, 0x60 );


    private final static int errorWidth = 17;
    private final static int errorHeight = 17;
    private static int[] errorPixels = {	// color model is AARRGGBB
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff
	};
    private static ImageProducer errorSource = null;

    public static Image errorIcon( Component comp )
	{
	if ( errorSource == null )
	    errorSource = new MemoryImageSource(
		errorWidth, errorHeight, errorPixels, 0, errorWidth );
	return comp.createImage( errorSource );
	}
    
    /// Medium red.
    public static final Color errorColor = new Color( 0xff, 0x70, 0x70 );


    private final static int busyWidth = 19;
    private final static int busyHeight = 21;
    private static int[] busyPixels = {		// color model is AARRGGBB
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0x00ffffff,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff
	};
    private static ImageProducer busySource = null;

    public static Image busyIcon( Component comp )
	{
	if ( busySource == null )
	    busySource = new MemoryImageSource(
		busyWidth, busyHeight, busyPixels, 0, busyWidth );
	return comp.createImage( busySource );
	}
    
    /// Pale turquoise.
    public static final Color busyColor = new Color( 0xb0, 0xff, 0xff );


    private final static int qmarkWidth = 17;
    private final static int qmarkHeight = 27;
    private static int[] qmarkPixels = {	// color model is AARRGGBB
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0xff000000,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000,
	0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0xff000000, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0xff000000, 0xff000000, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff
	};
    private static ImageProducer qmarkSource = null;

    public static Image qmarkIcon( Component comp )
	{
	if ( qmarkSource == null )
	    qmarkSource = new MemoryImageSource(
		qmarkWidth, qmarkHeight, qmarkPixels, 0, qmarkWidth );
	return comp.createImage( qmarkSource );
	}
    
    /// Pale green.
    public static final Color qmarkColor = new Color( 0xd0, 0xff, 0xd0 );


    private final static int brokenWidth = 20;
    private final static int brokenHeight = 19;
    private static int[] brokenPixels = {	// color model is AARRGGBB
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0x00ffffff,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,
	0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xffffffff,
	0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000,
	0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff00ff00, 0xff000000,
	0xff000000, 0xffff00ff, 0xffff00ff, 0xff000000, 0xffffffff,
	0xffffffff, 0xff000000, 0x00ffffff, 0x00ffffff, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff00ff00,
	0xff00ff00, 0xff000000, 0xff000000, 0xff00ff00, 0xff00ff00,
	0xff000000, 0xff000000, 0xffff00ff, 0xff000000, 0xffffffff,
	0xffffffff, 0xffffffff, 0xff000000, 0x00ffffff, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff00ff00,
	0xff000000, 0xff000000, 0xffff00ff, 0xff000000, 0xff00ff00,
	0xff00ff00, 0xff000000, 0xffff00ff, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0x00ffffff,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000,
	0xff000000, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff00ff00,
	0xff00ff00, 0xff000000, 0xffff00ff, 0xffff00ff, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xff000000, 0xff00ff00, 0xff00ff00,
	0xff000000, 0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xff000000, 0xff00ff00, 0xff00ff00, 0xff000000,
	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xff000000, 0xff00ff00, 0xff00ff00, 0xff000000, 0xff000000,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xff000000, 0xff00ff00, 0xff000000, 0xff000000, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xff000000, 0xff000000, 0xff000000, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xff000000, 0xff000000, 0xff000000, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xff000000, 0xff00ff00, 0xff00ff00, 0xff000000, 0xff000000,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xff000000, 0xff00ff00, 0xff000000, 0xff000000, 0xff000000,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xff000000, 0xff000000, 0xff000000, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff, 0xffff00ff,
	0xffff00ff, 0xffff00ff, 0xffff00ff, 0xff000000, 0xff000000,

	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,

	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000,
	};
    private static ImageProducer brokenSource = null;

    public static Image brokenIcon( Component comp )
	{
	if ( brokenSource == null )
	    brokenSource = new MemoryImageSource(
		brokenWidth, brokenHeight, brokenPixels, 0, brokenWidth );
	return comp.createImage( brokenSource );
	}
    

    /// Draw a centered image.
    public static void drawImage( Graphics graphics, Component comp, Image image )
	{
	Dimension d = comp.size();
	graphics.setColor( comp.getBackground() );
	graphics.fillRect( 0, 0, d.width, d.height );
	graphics.drawImage(
	    image,
	    ( d.width - image.getWidth( null ) ) / 2,
	    ( d.height - image.getHeight( null ) ) / 2,
	    null );
	}

    }
